unit ListList_Delphi_7U;
{$OPTIMIZATION ON}
{ Hierarchische Listenstrukturen am Beispiel (simulierter) Dateien und Ordner.
  SingleEntry = einzelne Datei, Basisklasse
  ListEntry = Verzeichnis, fhrt eine Liste mit Single- und ListEntry-Elementen

  Die Struktur wird per Zufallsgenerator aufgebaut, wobei die Wahrscheinlichkeit
  fr das Erzeugen neuer ListEntry-Elemente mit jeder zustzlichen Hierarchie-
  Ebene sinkt:
    if (Random(10*Level) < 1) then <New List, Rekursion mit Level+1>
     else <New Single Entry>
  Das Element der obersten Ebene ist eine Liste, entspricht dem Stammverzeichnis.

  Untersucht das Laufzeitverhalten von
  - Objektverwaltung (1 Mio Objekte)
  - Stringmanipulationen
  - Listen bzw. Arrays (24 Millionen Suchvorgnge)
  - Random-Generator
  - Stackverwaltung
}
interface

uses
  Windows, Messages, SysUtils, Variants, Classes, Graphics, Controls, Forms,
  Dialogs, StdCtrls, MMSystem;

type
  TSingleEntry = class  // einzelnes Element
    protected
      FIsList: Boolean;
    public
      EntryName: String;
      property IsList: Boolean read FIsList;
    end;

  TListEntry = class(TSingleEntry)   // Verzeichnis
  public
    EntryList: TList;  // TSingleEntry und TListEntry gemischt
    constructor Create;
    destructor Destroy; override;
  end;

type
  TLLForm = class(TForm)
    bCreateTree: TButton;  // Demo: Struktur mit 100 Elementen
    bBenchCreate: TButton; // 10000 Elemente, 100 mal aufgebaut
    ListBox1: TListBox;
    procedure bCreateTreeClick(Sender: TObject);
    procedure bBenchCreateClick(Sender: TObject);
  private
    procedure BuildBaseList(ECount: Integer);
    procedure ShowEntry(S: String);
    procedure ClearList;
  public
    EntryCount: Integer; // Runterzhler frs Anlegen der Struktur
    RootDir: TListEntry; // "Stammverzeichnis"
    procedure BuildListList(List: TListEntry; Level: Integer);
    function FindListEntry(List: TListEntry; SubStr: String): String;
    procedure PrintListList(List: TListEntry; Level: Integer);
  end;

var
  LLForm: TLLForm;

implementation
{$R *.dfm}

// Durchsucht die Struktur rekursiv und liefert den vollstndigen
// Pfad des ersten Vorkommens von SubStr zurck. Aufruf "von auen" mit RootDir
function TLLForm.FindListEntry(List: TListEntry; SubStr: String): String;
var x: Integer;
begin
  Result := '';
  // Einzelne Eintrge und Listennamen
  with List.EntryList do
    for x := 0 to Count-1 do
        // if Pos(SubStr, TSingleEntry(Items[x]).EntryName) <> 0 then
        if TSingleEntry(Items[x]).EntryName = SubStr then
        begin
          Result := TSingleEntry(Items[x]).EntryName;
          Break;
        end;
  // Rekursion fr Listen, falls noch nichts gefunden
  if Result = '' then
  begin
    with List.EntryList do
      for x := 0 to Count-1 do
        if TSingleEntry(Items[x]).IsList then
        begin
          Result := FindListEntry(TListEntry(Items[x]), SubStr);
          if Result <> '' then Break;
        end;
  end;

  if Result <> '' then Result := List.EntryName+'\'+Result;
end;

// Demo-Liste mit 100 Elementen, Anzeige
procedure TLLForm.bCreateTreeClick(Sender: TObject);
begin
  ClearList;
  ListBox1.Clear;
  BuildBaseList(100);
  PrintListList(RootDir, 1);
  ClearList;
end;

// Legt die Elementenzahl fest und erzeugt das Stammverzeichnis
procedure TLLForm.BuildBaseList(ECount: Integer);
begin
  EntryCount := ECount;
  while EntryCount > 0 do
    BuildListList(RootDir,1);
end;

procedure TLLForm.ClearList;
begin
  if RootDir <> nil then RootDir.Free;
  RootDir := TListEntry.Create;
  RootDir.EntryName := 'C:';
end;


// Listenaufbau (rekursiv)
procedure TLLForm.BuildListList(List: TListEntry; Level: Integer);
var x, y, LocalCount, NameLength: Integer;
    NewName: String;
    NewEntry: TSingleEntry;
begin
  // maximal 25 Eintrge pro Liste - nur das Stammverzeichnis
  // kann mehr haben
  LocalCount := Random(25)+1;
  for x := 1 to LocalCount do
  begin
    Dec(EntryCount);
    // zuflliger Name mit 1-8 Zeichen
    NameLength := Random(8)+1;
    NewName := '';
    for y := 1 to NameLength do
      NewName := NewName + Chr(Ord('a')+Random(26));

    // Einzelner Eintrag oder neue Liste? Die Wahrscheinlichkeit
    // fr neue Listen (und weitere Rekursion) sinkt mit zunehmender
    // Verschachelungstiefe
    if Random(10*Level) < 1 then
    begin  // new list
      NewEntry := TListEntry.Create;
      BuildListList(TListEntry(NewEntry), Level+1);
    end else
    begin
      NewEntry := TSingleEntry.Create;
    end;
    NewEntry.EntryName := NewName;

    List.EntryList.Add(NewEntry);

    if EntryCount <= 0 then Break;
  end;
end;

// Anzeige in der Listbox, auch fr Suchergebnisse
procedure TLLForm.ShowEntry(S: String);
begin
  ListBox1.Items.Add(S);
end;

// Ausgabe der Demo-Liste (100 Elemente)
procedure TLLForm.PrintListList(List: TListEntry; Level: Integer);
var x: Integer; Lead: String;
begin
  Lead := '';
  for x := 1 to Level-1 do Lead := Lead + '  ';
  ShowEntry(Lead+List.EntryName+' ###'); // Verzeichnisname
  Lead := Lead + '  ';  // Dateinamen eingerckt

  // Dateieintrge und Verzeichnisse, ungeordnet
  with List.EntryList do
    for x := 0 to Count-1 do
    begin
      if TSingleEntry(Items[x]).IsList
       then PrintListList(TListEntry(Items[x]), Level+1)
       else ShowEntry(Lead+TSingleEntry(Items[x]).EntryName);
    end;
end;

// 1 Million Objekte, 20 Millionen Suchvorgnge
procedure TLLForm.bBenchCreateClick(Sender: TObject);
var x, y: Integer;
    ConstructionTime, DestructionTime, FindTime: TDateTime;
    FullBench, Start, Meantime: TDateTime;

  function TStr(Cap: String;Time: TDateTime): String;
  var Temp: String;
  begin
    Temp := LongTimeFormat;
    LongTimeFormat := 'nn:ss:zzz';
    Result := Format('%-25s'#9'%s'#13#10, [Cap, TimeToStr(Time)]);
    LongTimeFormat := Temp;
  end;

begin
  ListBox1.Clear;
  ClearList;
  ConstructionTime := 0; DestructionTime := 0; FindTime := 0;
  Start := Now;
  for x := 1 to 10 do
  begin
    Meantime := Now;
    BuildBaseList(100000);
    ConstructionTime := ConstructionTime + (Now-Meantime);
    Meantime := Now;
    ShowEntry('Search a: ' + FindListEntry(RootDir, 'a'));
    ShowEntry('Search ax: ' + FindListEntry(RootDir, 'ax'));
    ShowEntry('Search axv: ' + FindListEntry(RootDir, 'axv'));
    ShowEntry('Search axve: ' + FindListEntry(RootDir, 'axve'));
    for y := 1 to 20 do
      FindListEntry(RootDir,'X'); // gibts nicht
    FindTime := FindTime + (Now-MeanTime);
    ListBox1.Update;
    MeanTime := Now;
    ClearList;  // in Delphi direkter Destruktoraufruf
    DestructionTime := DestructionTime+ (Now-Meantime);
  end;
  FullBench := Now-Start;

  ShowMessage(TStr('Time:',        FullBench)+
              TStr('Construction:',constructionTime)+
              TStr('Destruction:' ,destructionTime)+
              TStr('Find:',        findTime));

end;

{ TListEntry }
constructor TListEntry.Create;
begin
  FIsList := True;
  EntryList := TList.Create;
end;

// In Delphi ist ein Destruktor ntig, der die Listen durchluft
destructor TListEntry.Destroy;
var x: Integer;
begin
  with EntryList do
    for x := 0 to Count-1 do
      TSingleEntry(Items[x]).Free;
  EntryList.Free;
end;

begin
  Randomize;
end.
